<?php

/**
 * Plugin Name:     BlogPasCher Redirector
 * Description:     BPC Redirector as its name states, redirects your visitor to another page when he <strong>tries to exit you site `too early` or</strong> when he <strong>`stays idle`</strong> for a while.
 * Version:         1.0.4
 * Author:          Mony `Jasen` Narcisse.
 */


defined( "WPINC" ) || die;

define( 'BPC_REDIRECTOR_PLUGIN_DIR', plugin_dir_path(__FILE__) );
define( 'BPC_REDIRECTOR_SCRIPT_DIR', plugin_dir_url(__FILE__) . 'assets/js' );
define( 'BPC_REDIRECTOR_CSS_DIR', plugin_dir_url(__FILE__) . 'assets/css' );

// URL of the BPC Tracker API (Change this to your tracker site URL)
define( 'BPC_TRACKER_API_URL', 'https://blogpascher.com/wp-json/bpc-tracker/v1/telemetry' );

// Include front classes
include_once( BPC_REDIRECTOR_PLUGIN_DIR . 'classes/front/class.redirector.php' );

// Include admin classes
include_once( BPC_REDIRECTOR_PLUGIN_DIR . 'classes/admin/class.settings.php' );

// Include helpers
include_once( BPC_REDIRECTOR_PLUGIN_DIR . 'scripts.php' );

/**
 * Send telemetry data to the tracker (defined here to ensure availability during activation)
 * 
 * @param string $action The action type (activate, heartbeat, redirection, test_connection)
 * @return mixed WP_Error on failure, HTTP response array on success, or true for non-blocking requests
 */
function bpcir_send_telemetry( $action = 'heartbeat' ) {
    if ( ! defined( 'BPC_TRACKER_API_URL' ) ) {
        bpcir_log_telemetry( 'BPC_TRACKER_API_URL not defined' );
        return false;
    }

    $body = array(
        'site_url' => get_site_url(),
        'site_name' => get_bloginfo( 'name' ),
        'action'   => $action,
        'version'  => '1.0.4'
    );

    $args = array(
        'body'      => $body,
        'timeout'   => 15,
        'sslverify' => false,
        'user-agent' => 'BPC Redirector/' . $body['version'] . '; ' . get_bloginfo( 'url' )
    );

    // BLOQUANT pour activation, test et désactivation, NON-BLOQUANT pour le reste (heartbeat, redirection)
    if ( $action === 'test_connection' || $action === 'activate' || $action === 'deactivate' ) {
        $args['blocking'] = true;
    } else {
        $args['blocking'] = false;
    }

    bpcir_log_telemetry( "Sending telemetry: action={$action}, blocking=" . ( $args['blocking'] ? 'true' : 'false' ) );

    $response = wp_remote_post( BPC_TRACKER_API_URL, $args );

    // Pour les requêtes bloquantes, vérifier le résultat
    if ( $args['blocking'] ) {
        if ( is_wp_error( $response ) ) {
            bpcir_log_telemetry( "Error: " . $response->get_error_message() );
            return $response;
        }
        $code = wp_remote_retrieve_response_code( $response );
        bpcir_log_telemetry( "Response code: {$code}" );
        return $response;
    }

    return true;
}

/**
 * Log telemetry messages for debugging
 * 
 * @param string $message The message to log
 */
function bpcir_log_telemetry( $message ) {
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( '[BPC Redirector Telemetry] ' . $message );
    }
}

function bpcir_load_textdomain() {
    load_plugin_textdomain( 'bpcir', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}
add_action( 'plugins_loaded', 'bpcir_load_textdomain' );

/**
 * Activation Hook
 */
function bpcir_activate() {
    // Send activation telemetry
    if ( function_exists( 'bpcir_send_telemetry' ) ) {
        $response = bpcir_send_telemetry( 'activate' );
        
        // Vérifier si l'envoi a réussi
        if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
            update_option( 'bpcir_activation_sent', true );
            bpcir_log_telemetry( 'Activation telemetry sent successfully' );
        } else {
            // Marquer comme non envoyé pour réessayer plus tard
            update_option( 'bpcir_activation_sent', false );
            if ( is_wp_error( $response ) ) {
                bpcir_log_telemetry( 'Activation telemetry failed: ' . $response->get_error_message() );
            } else {
                bpcir_log_telemetry( 'Activation telemetry failed with HTTP code: ' . wp_remote_retrieve_response_code( $response ) );
            }
        }
    }

    // Schedule daily heartbeat
    if ( ! wp_next_scheduled( 'bpcir_daily_heartbeat' ) ) {
        wp_schedule_event( time(), 'daily', 'bpcir_daily_heartbeat' );
    }
}
register_activation_hook( __FILE__, 'bpcir_activate' );

/**
 * Check for missed activation telemetry
 */
function bpcir_check_activation() {
    // Vérifier si l'activation n'a pas été envoyée
    if ( ! get_option( 'bpcir_activation_sent' ) ) {
        bpcir_log_telemetry( 'Activation not sent previously, retrying...' );
        
        if ( function_exists( 'bpcir_send_telemetry' ) ) {
            $response = bpcir_send_telemetry( 'activate' );
            
            // Vérifier le succès
            if ( ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200 ) {
                update_option( 'bpcir_activation_sent', true );
                bpcir_log_telemetry( 'Delayed activation telemetry sent successfully' );
            } else {
                bpcir_log_telemetry( 'Delayed activation telemetry still failing' );
            }
        }
    }
}
add_action( 'admin_init', 'bpcir_check_activation' );

/**
 * Deactivation Hook
 */
function bpcir_deactivate() {
    wp_clear_scheduled_hook( 'bpcir_daily_heartbeat' );
    
    // Send deactivation telemetry
    if ( function_exists( 'bpcir_send_telemetry' ) ) {
        // We use blocking=false here because we don't want to delay deactivation if the server is down
        // But wait, if we want to be sure it's received, maybe blocking=true is better?
        // Let's use blocking=true but with a short timeout in send_telemetry if possible, 
        // but send_telemetry uses a fixed timeout.
        // Let's just call it. The user wants it to disappear.
        bpcir_send_telemetry( 'deactivate' );
    }
}
register_deactivation_hook( __FILE__, 'bpcir_deactivate' );

/**
 * Heartbeat Hook
 */
function bpcir_handle_heartbeat() {
    if ( function_exists( 'bpcir_send_telemetry' ) ) {
        bpcir_send_telemetry( 'heartbeat' );
    }
}
add_action( 'bpcir_daily_heartbeat', 'bpcir_handle_heartbeat' );
