<?php

defined( "WPINC" ) || die;

class SettingsPage {

    private $page_hook_suffix;

    private $default_options = array(
        "earlyLeave" => array(
            "cookie" => "bpcel",
            "cookieExp" => 30,
            "timeout" => 15,
            "eiZoneHeight" => 50,
            "zindex" => 999999
        ),
        "inactivity" => array(
            "cookie" => "bpcito",
            "cookieExp" => 30,
            "timeout" => 300
        ),
        "triggers" => array(
            "back_button" => 0,
            "scroll" => 0,
            "scroll_percentage" => 60,
            "copy" => 0
        ),
        "license" => array(
            "key" => "",
            "status" => "inactive"
        )
    );

    public function __construct() {
        
        $this->page_hook_suffix = "";

        add_action('admin_menu', array($this, 'register_options'));
        add_action('admin_menu', array($this, 'display_menu'));

        add_action( 'admin_enqueue_scripts', array($this, 'enqueue_repeater_field'));
        add_action( 'admin_footer', array($this, 'insert_repeater_template'));

        add_action( 'wp_ajax_bpcir_test_connection', array($this, 'test_connection') );
    }

    public function test_connection() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        if ( function_exists( 'bpcir_send_telemetry' ) ) {
            $response = bpcir_send_telemetry( 'test_connection' );
            
            if ( is_wp_error( $response ) ) {
                wp_send_json_error( $response->get_error_message() );
            } else {
                $code = wp_remote_retrieve_response_code( $response );
                $body = wp_remote_retrieve_body( $response );
                
                if ( $code === 200 ) {
                    wp_send_json_success( array( 'message' => 'Connection successful!', 'response' => $body ) );
                } else {
                    wp_send_json_error( 'HTTP Error: ' . $code . ' - ' . $body );
                }
            }
        } else {
            wp_send_json_error( 'Telemetry function not found.' );
        }
    }

    public function register_options() {
        // Register the settings
        register_setting(
            'bpcir_options_group',  /* Option Group */
            'bpcir_options',        /* Option Name */
            array($this, 'save_options')     /* Sanitize Callback */
        );
    }
    
    /**
     * Add Plugin menu as top level menu
     */
    public function display_menu() {
        $this->page_hook_suffix =  add_menu_page( 
            __( 'Redirector options', 'bpcir' ),
            __( 'BPC Redirector', 'bpcir' ),
            'manage_options',
            'redirector-options',
            array($this, 'display_menu_page'),
            'dashicons-randomize',
            80
        );
    }

    /**
     * Render options page
     */
    public function display_menu_page() {
        $bpcir_options = get_option( 'bpcir_options' );        
        
        $in_urls_links = isset($bpcir_options["inactivity"]["urls"]["link"]) ? $bpcir_options["inactivity"]["urls"]["link"] : array();
        $in_urls_timeouts = isset($bpcir_options["inactivity"]["urls"]["timeout"]) ? $bpcir_options["inactivity"]["urls"]["timeout"] : array();

        $in_urls_count = count($in_urls_links);

        

        include_once BPC_REDIRECTOR_PLUGIN_DIR . 'views/templates/bpcir_config_page.html.php';
    }

    /**
     * Include repeater template in footer
     */
    public function insert_repeater_template() {
        include_once BPC_REDIRECTOR_PLUGIN_DIR . 'assets/js/admin/repeater-field/template.html.php';
    }

    /**
     * Enqueue necessary js file for the repeater fields
     */
    public function enqueue_repeater_field() {
        wp_enqueue_script('bpc-repeater', "" , array(), false, true);
        wp_enqueue_style('bpc-admin-style', BPC_REDIRECTOR_CSS_DIR . '/admin-style.css');
    }


    public function save_options( $options ) {

        // var_dump($options); die;

        // `Early leaves` section fields 
        if( isset( $options['earlyLeave']["url"] ) && $options['earlyLeave']["url"] == "" ) {
            // Trigger error
            add_settings_error(
                'bpcir_options',
                'default_earlyLeave_url_empty',
                __( "Redirector Warning : Leaving early leave URL empty deactivates early leave feature.", 'bpcir' ),
                'info'
            );
        }
        
        if( isset( $options['earlyLeave']["cookie"] ) && $options['earlyLeave']["cookie"] == "" ) {
            $options['earlyLeave']["cookie"] = $this->default_options["earlyLeave"]["cookie"];
        }

        if( isset( $options['earlyLeave']["cookieExp"] ) && $options['earlyLeave']["cookieExp"] == ""  ) {
            $options['earlyLeave']["cookieExp"] = (int)$this->default_options["earlyLeave"]["cookieExp"];
        }

        if( isset( $options['earlyLeave']["timeout"] ) && 
            ($options['earlyLeave']["timeout"] == "") || (int)$options['earlyLeave']["timeout"] <= 0  ) {
            $options['earlyLeave']["timeout"] = (int)$this->default_options["earlyLeave"]["timeout"];
        }

        if( isset( $options['earlyLeave']["eiZoneHeight"] ) && 
            ($options['earlyLeave']["eiZoneHeight"] == "") || (int)$options['earlyLeave']["eiZoneHeight"] <= 0  ) {
            $options['earlyLeave']["eiZoneHeight"] = (int)$this->default_options["earlyLeave"]["eiZoneHeight"];
        }

        if( isset( $options['earlyLeave']["zindex"] ) && 
            ($options['earlyLeave']["zindex"] == "") || (int)$options['earlyLeave']["zindex"] <= 0  ) {
            $options['earlyLeave']["zindex"] = (int)$this->default_options["earlyLeave"]["zindex"];
        }
        
        // `Inactivity` section fields
        if( isset( $options['inactivity']["url"] ) && $options['inactivity']["url"] == "" ) {
            // Trigger error
            add_settings_error(
                'bpcir_options',
                'default_inactivity_url_empty',
                __( "Redirector Warning : Leaving inactivity URL empty deactivates inactivity feature.", 'bpcir' ),
                'info'
            );
        }

        // TODO : Sanitize urls fields here to respect their types

        if( isset( $options['inactivity']["cookie"] ) && $options['inactivity']["cookie"] == "" ) {
            $options['inactivity']["cookie"] = $this->default_options["inactivity"]["cookie"];
        }

        if( isset( $options['inactivity']["cookieExp"] ) && $options['inactivity']["cookieExp"] == "") {
            $options['inactivity']["cookieExp"] = (int)$this->default_options["inactivity"]["cookieExp"];
        }

        if( isset( $options['inactivity']["timeout"] ) && 
            ($options['inactivity']["timeout"] == "") || (int)$options['inactivity']["timeout"] <= 0  ) {
            $options['inactivity']["timeout"] = (int)$this->default_options["inactivity"]["timeout"];
        }        

        $links = isset($options["inactivity"]["urls"]["link"]) ? $options["inactivity"]["urls"]["link"] : array();
        $timeouts = isset($options["inactivity"]["urls"]["timeout"]) ? $options["inactivity"]["urls"]["timeout"] : array();

        foreach($links as $key => $link) {
            if($link == "") {
                $options["inactivity"]["urls"]["link"][$key] = "";
                add_settings_error(
                    'bpcir_options',
                    'secondary_inactivity_url_empty',
                    __( "Redirector warning : Secondary links should not be empty. Redirection will not occur.", 'bpcir' ),
                    'warning'
                );
            }
        }

        foreach($timeouts as $key => $timeout) {            
            if(! is_numeric($timeout)) {
                $options["inactivity"]["urls"]["timeout"][$key] = $options['inactivity']["timeout"];
                add_settings_error(
                    'bpcir_options',
                    'secondary_inactivity_url_empty',
                    __( "Redirector warning : Link timeout should be a number. Setting default timeout.", 'bpcir' ),
                    'warning'
                );
            }
        }


        if($add_success_notice) {
            add_settings_error(
                'bpcir_options',
                'default_inactivity_url_empty',
                __( "Redirector : Settings saved.", 'bpcir' ),
                'updated'
            );
        }

        // Triggers sanitization
        if ( isset( $options['triggers'] ) ) {
            $options['triggers']['back_button'] = isset( $options['triggers']['back_button'] ) ? 1 : 0;
            $options['triggers']['scroll'] = isset( $options['triggers']['scroll'] ) ? 1 : 0;
            $options['triggers']['copy'] = isset( $options['triggers']['copy'] ) ? 1 : 0;
            
            $scroll_pct = isset( $options['triggers']['scroll_percentage'] ) ? (int) $options['triggers']['scroll_percentage'] : 60;
            if ( $scroll_pct < 0 ) $scroll_pct = 0;
            if ( $scroll_pct > 100 ) $scroll_pct = 100;
            $options['triggers']['scroll_percentage'] = $scroll_pct;
        } else {
            $options['triggers'] = array(
                'back_button' => 0,
                'scroll' => 0,
                'copy' => 0,
                'scroll_percentage' => 60
            );
        }

        // License Validation
        if ( isset( $options['license']['key'] ) && ! empty( $options['license']['key'] ) ) {
            $license_key = sanitize_text_field( $options['license']['key'] );
            
            // Only validate if key changed or status is not active
            $old_options = get_option( 'bpcir_options' );
            $old_key = isset( $old_options['license']['key'] ) ? $old_options['license']['key'] : '';
            $old_status = isset( $old_options['license']['status'] ) ? $old_options['license']['status'] : 'inactive';

            if ( $license_key !== $old_key || $old_status !== 'valid' ) {
                // Call API to validate
                if ( defined( 'BPC_TRACKER_API_URL' ) ) {
                    $api_url = str_replace( '/telemetry', '/validate-license', BPC_TRACKER_API_URL );
                    
                    $response = wp_remote_post( $api_url, array(
                        'body' => array(
                            'license_key' => $license_key,
                            'site_url' => get_site_url()
                        ),
                        'timeout' => 15
                    ) );

                    if ( is_wp_error( $response ) ) {
                        add_settings_error( 'bpcir_options', 'license_error', __( 'License validation failed: Connection error.', 'bpcir' ) );
                        $options['license']['status'] = 'inactive';
                    } else {
                        $body = wp_remote_retrieve_body( $response );
                        $data = json_decode( $body, true );

                        if ( isset( $data['status'] ) && $data['status'] === 'valid' ) {
                            $options['license']['status'] = 'valid';
                            add_settings_error( 'bpcir_options', 'license_valid', __( 'License activated successfully!', 'bpcir' ), 'updated' );
                        } else {
                            $options['license']['status'] = 'inactive';
                            $msg = isset( $data['message'] ) ? $data['message'] : __( 'Invalid license key.', 'bpcir' );
                            add_settings_error( 'bpcir_options', 'license_invalid', $msg );
                        }
                    }
                }
            } else {
                // Keep existing status if key hasn't changed and was valid
                $options['license']['status'] = $old_status;
            }
        } else {
            $options['license']['status'] = 'inactive';
        }

        return $options;
    }
}

$ir_settings_page = new SettingsPage();